/*
 * Copyright (C) 2016 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

//******************************************************************************
// Version history:
// 1.0 07/17             Initial version. (Nima Eskandari)
// 1.1 07/17             Added Comments. (Nima Eskandari)
//----------------------------------------------------------------------------
//   Designed 2017 by Texas Instruments
//
//   Nima Eskandari
//   Texas Instruments Inc.
//   August 2017
//   Built with CCS Version: Code Composer Studio v7
//******************************************************************************

#include <stdio.h>
#include <stdbool.h>
#include <stddef.h>
#include <unistd.h>
#include <time.h>
#include <Board.h>

#include <ti/sysbios/BIOS.h>
#include <xdc/runtime/Types.h>

#include <ti/drivers/timer/GPTimerCC26XX.h>
#include <semaphore.h>

#include <config.h>
#include <debug.h>

GPTimerCC26XX_Handle delayTimer;
GPTimerCC26XX_Params timerParams;
Types_FreqHz  freq;
sem_t timerSemaphore;

uint32_t counter = 0;

void timerCallback(GPTimerCC26XX_Handle handle, GPTimerCC26XX_IntMask interruptMask)
{
    counter --;
    if (counter <= 0)
        sem_post(&timerSemaphore);
}

void InitializeDelayTimer()
{
    BIOS_getCpuFreq(&freq);
    sem_init(&timerSemaphore, 0, 0);
    GPTimerCC26XX_Params_init(&timerParams);
    timerParams.width          = GPT_CONFIG_16BIT;
    timerParams.mode           = GPT_MODE_PERIODIC_UP;
    timerParams.debugStallMode = GPTimerCC26XX_DEBUG_STALL_OFF;
}

//*****************************************************************************
// Get delay in milli seconds *************************************************
// time: The number of milliseconds to delay **********************************
//*****************************************************************************
void MsDelay(uint32_t time)
{
    if (time <= 0)
    {
        return;
    }
    else if (time <= 2)
    {
        counter = 1;
    }
    else
    {
        counter = time - 1;
    }
    delayTimer = GPTimerCC26XX_open(Board_GPTIMER0A, &timerParams);
    if(delayTimer == NULL) {

    }
    GPTimerCC26XX_Value loadVal = (48 * 1000) - 1 ; //47999
    GPTimerCC26XX_setLoadValue(delayTimer, loadVal);
    GPTimerCC26XX_registerInterrupt(delayTimer, timerCallback, GPT_INT_TIMEOUT);

    GPTimerCC26XX_start(delayTimer);
    sem_wait(&timerSemaphore);
    GPTimerCC26XX_unregisterInterrupt(delayTimer);
    GPTimerCC26XX_stop(delayTimer);
    GPTimerCC26XX_close(delayTimer);
}

//*****************************************************************************
// Get delay in micro seconds *************************************************
// time: Number of micro seconds to delay *************************************
//*****************************************************************************
void usDelay(uint32_t time)
{
    if (time <= 0)
        return;

    if (time <= 200)
    {
        counter = 1;
    }
    else
    {
        counter = time / 100 - 1;
    }


    delayTimer = GPTimerCC26XX_open(Board_GPTIMER0A, &timerParams);
    if(delayTimer == NULL) {

    }
    GPTimerCC26XX_Value loadVal = (48 * 100) - 1 ; //4799
    GPTimerCC26XX_setLoadValue(delayTimer, loadVal);
    GPTimerCC26XX_registerInterrupt(delayTimer, timerCallback, GPT_INT_TIMEOUT);

    GPTimerCC26XX_start(delayTimer);
    sem_wait(&timerSemaphore);
    GPTimerCC26XX_unregisterInterrupt(delayTimer);
    GPTimerCC26XX_stop(delayTimer);
    GPTimerCC26XX_close(delayTimer);
}

//*****************************************************************************
// Print an array to console **************************************************
// dataArray: The data array to print *****************************************
// size: The size of the dataArray ********************************************
//*****************************************************************************
void printArray(uint8_t * dataArray, uint16_t size)
{
	int i = 0;
	for (i = 0; i < size; i++)
	{
	    Debug_UART_printStringWithNumber("0x%X, ", dataArray[i]);
	}
	Debug_UART_printString("\r\n");
}

//*****************************************************************************
// Gets the current time ******************************************************
//*****************************************************************************
int GetTime()
{
    return (int)time(NULL);
}
